/*
 * DailyRollingFileAppender.cpp
 *
 * See the COPYING file for the terms of usage and distribution.
 */

#include "PortabilityImpl.hh"
#ifdef LOG4CPP_HAVE_IO_H
#    include <io.h>
#endif
#ifdef LOG4CPP_HAVE_UNISTD_H
#    include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <log4cpp/DailyRollingFileAppender.hh>
#include <log4cpp/Category.hh>
#include <log4cpp/FactoryParams.hh>
#include <memory>
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <iostream>

#ifdef LOG4CPP_HAVE_SSTREAM
#include <sstream>
#include <iomanip>
#endif

#ifndef WIN32    // only available on Win32
#include <dirent.h>
#endif

#if (defined (WIN32))
#include <winsock.h>
#include <direct.h>
#include <process.h>
#include <shlobj.h>
#define snprintf _snprintf
#else
#include <arpa/inet.h>
#include <sys/stat.h>
#include <pwd.h>
#include <unistd.h>
#endif

namespace MVlog4cpp {

    // modify default log keep day from 30 to 5
    unsigned int DailyRollingFileAppender::maxDaysToKeepDefault = 5;
	std::string DailyRollingFileAppender::logsPath = "";

    DailyRollingFileAppender::DailyRollingFileAppender(const std::string& name,
                                             const std::string& fileName, 
                                             unsigned int maxDaysToKeep,
                                             bool append,
											 mode_t mode,
											 std::string logPath) :
		FileAppender(name, changeFileName(name, fileName, logPath), append, mode),
        _maxDaysToKeep(maxDaysToKeep != 0 ? maxDaysToKeep : maxDaysToKeepDefault){
        struct stat statBuf;
        int res;
        time_t t;

        // obtain last modification time
        res = ::stat(fileName.c_str(), &statBuf);
        if (res < 0) {
            t = time(NULL);
        } else {
            t = statBuf.st_mtime;
        }
#ifndef WIN32    // only available on Win32
        localtime_r(&t, &_logsTime);
#else
        localtime_s(&_logsTime, &t);
#endif

		// init : clean old log which is over max keep days
		cleanOldLog();
    }

    void DailyRollingFileAppender::setMaxDaysToKeep(unsigned int maxDaysToKeep) {
        _maxDaysToKeep = maxDaysToKeep;
    }

    unsigned int DailyRollingFileAppender::getMaxDaysToKeep() const {
        return _maxDaysToKeep;
    }

    void DailyRollingFileAppender::rollOver()
    {
        std::ostringstream filename_s;
        int res_close = ::close(_fd);
        if (res_close != 0) {
            std::cerr << "Error closing file " << _fileName << std::endl;
        }

        // modify log filename to the format like : time.pid.filename.log
        int npos = _fileName.rfind(".", _fileName.length());
        std::string strTemp = _fileName.substr(0, npos);

        npos = strTemp.rfind(".", strTemp.length());
        strTemp = strTemp.substr(npos + 1);
        strTemp = strTemp.append(".log");

        time_t t = time(NULL);
#ifndef WIN32
        localtime_r(&t, &_logsTime);
        filename_s << generatePath() << _logsTime.tm_year + 1900 << "-" 
            << std::setfill('0') << std::setw(2) << _logsTime.tm_mon + 1 << "-"
            << std::setw(2) << _logsTime.tm_mday 
            << "." << getpid()
            << "." << strTemp << std::ends;

#else
		std::string oldFileName = FileAppender::getFileName();
		oldFileName = oldFileName.substr(0, oldFileName.find_last_of("\\")+1);
        localtime_s(&_logsTime, &t);
		filename_s << oldFileName << _logsTime.tm_year + 1900 << "-"
            << std::setfill('0') << std::setw(2) << _logsTime.tm_mon + 1 << "-"
            << std::setw(2) << _logsTime.tm_mday 
            << "." << GetCurrentProcessId()
            << "." << strTemp << std::ends;
#endif

        const std::string newFileName = filename_s.str();

        _fd = ::open(newFileName.c_str(), _flags, _mode);
        if (_fd == -1) {
            std::cerr << "Error opening file " << newFileName << std::endl;
        }

		cleanOldLog();
    }

    void DailyRollingFileAppender::_append(const MVlog4cpp::LoggingEvent &event)
    {
        struct tm now;
        time_t t = time(NULL);

#ifndef WIN32    // only available on Win32
        bool timeok = localtime_r(&t, &now) != NULL;
#else
        bool timeok = localtime_s(&now, &t) == 0;
#endif
        if (timeok) {
            if ((now.tm_mday != _logsTime.tm_mday) ||
                (now.tm_mon != _logsTime.tm_mon) ||
                (now.tm_year != _logsTime.tm_year)) {
                rollOver();
                _logsTime = now;
            }
        }
        MVlog4cpp::FileAppender::_append(event);
    }

   std::auto_ptr<Appender> create_daily_roll_file_appender(const FactoryParams& params)
   {
	   std::string name, filename, logsPath;
      bool append = true;
      mode_t mode = 664;
      unsigned int max_days_keep = 0;
	  params.get_for("daily roll file appender").required("name", name)("filename", filename)("logsPath", logsPath)("max_days_keep", max_days_keep)
                                          .optional("append", append)("mode", mode);

	  return std::auto_ptr<Appender>(new DailyRollingFileAppender(name, filename, max_days_keep, append, mode, logsPath));
   }

   std::string DailyRollingFileAppender::changeFileName(const std::string& name, const std::string& fileName, std::string& logPath)
   {
	   logsPath = logPath;
	   
       // user do not identify the filename
       std::string strTemp = fileName;
       if (0 == strTemp.length())
       {
           strTemp = name + ".log";
       }
       else
       {
           // filename lack of postfix ".log"
           if (fileName.find(".") == std::string::npos)
           {
               strTemp = fileName + ".log";
           }
       }

       std::ostringstream filename_s;
       time_t t = time(NULL);

#ifndef WIN32
       localtime_r(&t, &_logsTime);
#else
       localtime_s(&_logsTime, &t);
#endif

#ifndef WIN32
       filename_s << generatePath() << _logsTime.tm_year + 1900 << "-" 
           << std::setfill('0') << std::setw(2) << _logsTime.tm_mon + 1 << "-"
           << std::setw(2) << _logsTime.tm_mday 
           << "." << getpid()
           << "." << strTemp << std::ends;
#else
       filename_s << generatePath() << _logsTime.tm_year + 1900 << "-" 
           << std::setfill('0') << std::setw(2) << _logsTime.tm_mon + 1 << "-"
           << std::setw(2) << _logsTime.tm_mday 
           << "." << GetCurrentProcessId()
           << "." << strTemp << std::ends;
#endif


       return filename_s.str();
   }

   std::string DailyRollingFileAppender::generatePath()
   {
       std::string strFilePath("");

#if (defined (WIN32))

       // ϵͳ/Users/û/application/
     //  char windowsDir[MAX_PATH] = { 0, };
      // (void)GetWindowsDirectory(windowsDir, MAX_PATH);

     //  char userName[MAX_PATH] = { 0, };

     //  DWORD userSize = MAX_PATH;
     //  (void)GetUserName(userName, &userSize);

       char moduleFileName[MAX_PATH] = { 0, };
       (void)GetModuleFileName(NULL, moduleFileName, MAX_PATH);

       char* applicationName = strrchr(moduleFileName, '\\');
       char* pEnd = strrchr(applicationName, '.');
       if (pEnd)
       {
           *pEnd = '\0';
       }

       if (NULL != getenv("USERPROFILE"))
       {
           //strFilePath.append(windowsDir).append("\\..").append(getenv("USERPROFILE")).append("\\MVSDK\\");
           //(void)_mkdir(strFilePath.c_str());
		   strFilePath.append(getenv("USERPROFILE")).append("\\MVSDK\\");
           strFilePath.append(applicationName).append("\\");
       }
       else
       {
           char szPath[MAX_PATH] = { 0 };
           SHGetFolderPath(NULL, CSIDL_PERSONAL | CSIDL_FLAG_CREATE, NULL, 0, szPath);
		   strFilePath.append(szPath).append("\\..\\").append("\\MVSDK\\");
		   strFilePath.append(applicationName).append("\\");
       }

	   if (!logsPath.empty())
	   {
		   strFilePath.clear();
		   strFilePath.append(logsPath.c_str());
		   strFilePath.append(applicationName).append("\\");
	   }

       //(void)_mkdir(strFilePath.c_str());
	   SHCreateDirectoryEx(NULL,strFilePath.c_str(),NULL);
#elif defined (__ANDROID__)
	   do
	   {
		   std::ostringstream command_s;
		   command_s << "cat /proc/" << getpid() << "/cmdline" << std::ends;
		   char readBuffer[250];
		   memset(readBuffer, 0, sizeof(readBuffer));
		   FILE *file_read = popen(command_s.str().c_str(), "r");
		   if(file_read == NULL)
		   {
			   break;
		   }

		   if(fread(readBuffer, sizeof(char), sizeof(readBuffer), file_read) == 0)
		   {
			   pclose(file_read);
			   break;
		   }
		   pclose(file_read);

		   strFilePath.append("/data/data/").append(std::string(readBuffer)).append("/");
	   }while(false);

	   //Ƿjavaûа,д̶λ/data/local/tmp/MVSDK/
	   if((strFilePath.length() == 0) || (access(strFilePath.c_str(), 0) < 0))
	   {
		   strFilePath = std::string("/data/local/tmp/MVSDK/");
	   }

	   (void)mkdir(strFilePath.c_str(), (mode_t)0777);
#else
       // /home/û/application/
       struct passwd* pwd;
       pwd = getpwuid(getuid());
       if(NULL == pwd)
       {
           return "";
       }

       char path[260];
       size_t len = 260;
       int rsltLen = readlink("/proc/self/exe", path, len);
       if(rsltLen <= 0)
       {
           return "";
       }

       path[rsltLen] = '\0';
       char* path_end = strrchr(path, '/');
       if(NULL == path_end)
       {
           return "";
       }

       ++path_end;

       strFilePath.append("/var/log/MVSDK/");

       static const mode_t mode = 0755;
       (void)mkdir(strFilePath.c_str(), mode);

       strFilePath.append(path_end).append("/");
       (void)mkdir(strFilePath.c_str(), mode);

#endif
       return std::string(strFilePath);

   }

   void DailyRollingFileAppender::cleanOldLog()
   {
	   std::string logPath = generatePath();
	   std::string logExt(".log");
	   const time_t oldest = time(NULL) - _maxDaysToKeep * 60 * 60 * 24;

#ifndef WIN32 
#define PATHDELIMITER "/" 
#else 
#define PATHDELIMITER "\\"
#endif

#ifndef WIN32
	   struct dirent **entries;
	   int nentries = scandir(logPath.c_str(), &entries, 0, alphasort);
	   if (nentries < 0)
		   return;
	   for (int i = 0; i < nentries; i++) {
		   struct stat statBuf;
		   const std::string fullfilename = logPath + PATHDELIMITER + entries[i]->d_name;
		   int res = ::stat(fullfilename.c_str(), &statBuf);
		   if ((res == -1) || (!S_ISREG(statBuf.st_mode))) {
			   free(entries[i]);
			   continue;
		   }
		   if (statBuf.st_mtime < oldest && strstr(entries[i]->d_name, logExt.c_str())) {
			   //std::cout << " Deleting " << fullfilename.c_str() << std::endl;
			   ::unlink(fullfilename.c_str());
		   }
		   free(entries[i]);
	   }
	   free(entries);
#else
	   HANDLE hFind = INVALID_HANDLE_VALUE;
	   WIN32_FIND_DATA ffd;
	   const std::string pattern = logPath + PATHDELIMITER + "*" + logExt;

	   hFind = FindFirstFile(pattern.c_str(), &ffd);
	   if (hFind != INVALID_HANDLE_VALUE) {
		   do {
			   struct stat statBuf;
			   const std::string fullfilename = logPath + PATHDELIMITER + ffd.cFileName;
			   int res = ::stat(fullfilename.c_str(), &statBuf);

			   if (res != -1 && statBuf.st_mtime < oldest && !(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
				   //std::cout << "Deleting " << fullfilename << "\n";
				   ::unlink(fullfilename.c_str());
			   }
		   } while (FindNextFile(hFind, &ffd) != 0);

		   if (GetLastError() != ERROR_NO_MORE_FILES) {
			   // [XXX] some kind of error happened
		   }
		   FindClose(hFind);
		   hFind = INVALID_HANDLE_VALUE;
	   }
#endif
   }

}
