/*
 * StringQueueAppender.cpp
 *
 * Copyright 2000, LifeLine Networks BV (www.lifeline.nl). All rights reserved.
 * Copyright 2000, Bastiaan Bakker. All rights reserved.
 *
 * See the COPYING file for the terms of usage and distribution.
 */

#include "PortabilityImpl.hh"
#include <log4cpp/StringQueueAppender.hh>
#include<exception>
#ifdef LOG4CPP_HAVE_SSTREAM
#include <sstream>
#include <iomanip>
#endif

#ifdef WIN32
#include <shlobj.h>
#define snprintf _snprintf
#else
#include <pthread.h>
#include <pwd.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#endif

bool errorThread = false;

namespace MVlog4cpp {

	StringQueueAppender::StringQueueAppender(const std::string& name, const std::string& fileName, bool append) :
		LayoutAppender(name),
		_flags("a"),
		_exitThread(true){
		if (!append)
			_flags = "w";
		_fileName = changeFileName(name, fileName);
		_file = fopen(_fileName.c_str(), _flags.c_str());
#ifdef WIN32
		try{
			_wThread = std::thread(&StringQueueAppender::threadFunction, this);
		}
		catch (const std::system_error &e) {
			errorThread = true;
		}
		//ж߳Ƿ񻹴
		_threadId = _wThread.get_id();
#else
		int ret = 0;
		pthread_mutex_init(&_lMtx, NULL);
		ret = pthread_create(&_lThread, NULL, StringQueueAppender::threadFunction, this);
		if (ret != 0)
		{
			errorThread = true;
		}
#endif
    }
    
    StringQueueAppender::~StringQueueAppender() {
        close();
    }

    void StringQueueAppender::close() {
		_exitThread = false;
#ifdef WIN32
		if (_wThread.joinable())
		{
			_wThread.join();
		}
#else
		pthread_join(_lThread, NULL);
		pthread_mutex_destroy(&_lMtx);
#endif
		if (_file != NULL)
			fclose(_file);
    }

    void StringQueueAppender::_append(const LoggingEvent& event) {
		if (!errorThread)
		{
#ifdef WIN32
			std::unique_lock<std::mutex> lock(_wMtx);
			_queue.push(_getLayout().format(event));
#else
			pthread_mutex_lock(&_lMtx);
			_queue.push(_getLayout().format(event));
			pthread_mutex_unlock(&_lMtx);
#endif
		}
    }

    bool StringQueueAppender::reopen() {
		if (_fileName != "")
		{
			FILE* hFile = NULL;
			hFile = fopen(_fileName.c_str(), _flags.c_str());
			if (hFile != NULL)
			{
				if (_file != NULL)
					fclose(_file);
				_file = hFile;
#ifdef WIN32
				if (_wThread.get_id() != _threadId)
				{
					_exitThread = true;
					_wThread = std::thread(&StringQueueAppender::threadFunction, this);
					_threadId = _wThread.get_id();
				}
#else
				int ret = pthread_kill(_lThread, 0);
				if (ret == ESRCH)
				{
					_exitThread = true;
					pthread_create(&_lThread, NULL, StringQueueAppender::threadFunction, this);
				}
#endif
				return true;
			}
		}
        return false;
    }      

    std::queue<std::string>& StringQueueAppender::getQueue() {
        return _queue;
    }

    const std::queue<std::string>& StringQueueAppender::getQueue() const {
        return _queue;
    }

    size_t StringQueueAppender::queueSize() const {
        return getQueue().size();
    }

    std::string StringQueueAppender::popMessage() {        
        std::string message;

        if (!_queue.empty()) {
            message = _queue.front();
#ifdef WIN32
			std::unique_lock<std::mutex> lock(_wMtx);
			_queue.pop();
#else
			pthread_mutex_lock(&_lMtx);
			_queue.pop();
			pthread_mutex_unlock(&_lMtx);
#endif
        }

        return message;
    }

	void* StringQueueAppender::threadFunction(void* argv)
	{
		StringQueueAppender* s = (StringQueueAppender*)argv;
		if (s->_file == NULL) return NULL;
		std::string message;
		while (s->_exitThread)
		{
			while (!s->_queue.empty())
			{
				message = s->_queue.front();
				if (fwrite(message.data(), message.length(), 1, s->_file)) 
				{
#ifdef WIN32
					std::unique_lock<std::mutex> lock(s->_wMtx);
					s->_queue.pop();
#else
					pthread_mutex_lock(&s->_lMtx);
					s->_queue.pop();
					pthread_mutex_unlock(&s->_lMtx);			
#endif
				}
			}
#ifdef WIN32
			Sleep(1);
#else
			usleep(1000);
#endif
		}
		return NULL;
	}

	std::string StringQueueAppender::changeFileName(const std::string& name, const std::string& fileName)
	{
		// user do not identify the filename
		std::string strTemp = fileName;
		if (0 == strTemp.length())
		{
			strTemp = name + ".log";
		}
		else
		{
			// filename lack of postfix ".log"
			if (fileName.find(".") == std::string::npos)
			{
				strTemp = fileName + ".log";
			}
		}

		std::ostringstream filename_s;
		time_t t = time(NULL);

#ifndef WIN32
		localtime_r(&t, &_logsTime);
#else
		localtime_s(&_logsTime, &t);
#endif

#ifndef WIN32
		filename_s << generatePath() << _logsTime.tm_year + 1900 << "-"
			<< std::setfill('0') << std::setw(2) << _logsTime.tm_mon + 1 << "-"
			<< std::setw(2) << _logsTime.tm_mday
			<< "." << getpid()
			<< "." << strTemp << std::ends;
#else
		filename_s << generatePath() << _logsTime.tm_year + 1900 << "-"
			<< std::setfill('0') << std::setw(2) << _logsTime.tm_mon + 1 << "-"
			<< std::setw(2) << _logsTime.tm_mday
			<< "." << GetCurrentProcessId()
			<< "." << strTemp << std::ends;
#endif
		return filename_s.str();
	}

	std::string StringQueueAppender::generatePath()
	{
		std::string strFilePath("");

#if (defined (WIN32))

		// ϵͳ/Users/û/application/
		//  char windowsDir[MAX_PATH] = { 0, };
		// (void)GetWindowsDirectory(windowsDir, MAX_PATH);

		//  char userName[MAX_PATH] = { 0, };

		//  DWORD userSize = MAX_PATH;
		//  (void)GetUserName(userName, &userSize);

		char moduleFileName[MAX_PATH] = { 0, };
		(void)GetModuleFileName(NULL, moduleFileName, MAX_PATH);

		char* applicationName = strrchr(moduleFileName, '\\');
		char* pEnd = strrchr(applicationName, '.');
		if (pEnd)
		{
			*pEnd = '\0';
		}

		if (NULL != getenv("USERPROFILE"))
		{
			//strFilePath.append(windowsDir).append("\\..").append(getenv("USERPROFILE")).append("\\MVSDK\\");
			//(void)_mkdir(strFilePath.c_str());
			strFilePath.append(getenv("USERPROFILE")).append("\\MVSDK\\");
			strFilePath.append(applicationName).append("\\");
		}
		else
		{
			char szPath[MAX_PATH] = { 0 };
			SHGetFolderPath(NULL, CSIDL_PERSONAL | CSIDL_FLAG_CREATE, NULL, 0, szPath);
			strFilePath.append(szPath).append("\\..\\").append("\\MVSDK\\");
			strFilePath.append(applicationName).append("\\");
		}
		//(void)_mkdir(strFilePath.c_str());
		SHCreateDirectoryEx(NULL, strFilePath.c_str(), NULL);
#elif defined (__ANDROID__)
		do
		{
			std::ostringstream command_s;
			command_s << "cat /proc/" << getpid() << "/cmdline" << std::ends;
			char readBuffer[250];
			memset(readBuffer, 0, sizeof(readBuffer));
			FILE *file_read = popen(command_s.str().c_str(), "r");
			if (file_read == NULL)
			{
				break;
			}

			if (fread(readBuffer, sizeof(char), sizeof(readBuffer), file_read) == 0)
			{
				pclose(file_read);
				break;
			}
			pclose(file_read);

			strFilePath.append("/data/data/").append(std::string(readBuffer)).append("/");
		} while (false);

		if ((strFilePath.length() == 0) || (access(strFilePath.c_str(), 0) < 0))
		{
			strFilePath = std::string("/data/data/MVSDK/");
			(void)mkdir(strFilePath.c_str(), (mode_t)0755);
		}
#else
		// /home/û/application/
		struct passwd* pwd;
		pwd = getpwuid(getuid());
		if (NULL == pwd)
		{
			return "";
		}

		char path[260];
		size_t len = 260;
		int rsltLen = readlink("/proc/self/exe", path, len);
		if (rsltLen <= 0)
		{
			return "";
		}

		path[rsltLen] = '\0';
		char* path_end = strrchr(path, '/');
		if (NULL == path_end)
		{
			return "";
		}

		++path_end;

		strFilePath.append("/var/log/MVSDK/");

		static const mode_t mode = 0755;
		(void)mkdir(strFilePath.c_str(), mode);

		strFilePath.append(path_end).append("/");
		(void)mkdir(strFilePath.c_str(), mode);
#endif
		return std::string(strFilePath);
	}
}


